# Keycloak Helm Chart

A Helm chart to deploy [Keycloak](https://www.keycloak.org/) in a non-operator manner on Kubernetes.

This chart provides a flexible and customizable way to set up Keycloak, including an optional internal PostgreSQL database, support for custom themes, and mechanisms for realm importation or database provisioning.

## Prerequisites

*   Kubernetes 1.12+
*   Helm 3.2.0+
*   An Ingress controller (e.g., [NGINX Ingress Controller](https://kubernetes.github.io/ingress-nginx/)) if `ingress.enabled` is `true`.

## Installation

To install the chart with the release name `my-release` into the `keycloak` namespace:

```bash
helm install my-release ./keycloak-chart --namespace keycloak --create-namespace
```

### Example Installation with Ingress

For a more realistic setup, you might want to enable ingress to expose Keycloak outside the cluster.

```bash
helm install my-release ./keycloak-chart \
  --namespace keycloak \
  --create-namespace \
  --set ingress.enabled=true \
  --set ingress.hosts[0].host=keycloak.example.com \
  --set ingress.hosts[0].paths[0].path=/ \
  --set keycloak.adminUser=admin \
  --set keycloak.adminPassword=mysecretpassword
```

### Local Development

For local development, you can use the provided `Taskfile.yaml`. First, create a `.env` file:

```bash
# .env
KUBE_CONFIG=k3d.cluster.config # Path to your kubeconfig file
GL_SLUG=local-testing          # Suffix for the namespace (kc-local-testing)
REVIEW_DOMAIN=docker.localhost # Domain for review apps
KEYCLOAK_HOSTNAME=kc.docker.localhost # Hostname for Keycloak
VALUE_FILE=kc-values.yaml      # Custom values file to use
```

Then, run the installation task:

```bash
task helm:install:local
```

## Accessing Keycloak

Once the chart is deployed, you can access the Keycloak UI.

If you enabled ingress, the URL will be printed in the installation notes. You can also find it with:

```bash
echo "https://$(kubectl get ingress -n keycloak my-release-keycloak-helm -o jsonpath='{.spec.rules[0].host}')"
```

If you are not using ingress, you can access Keycloak via port-forwarding:

```bash
export POD_NAME=$(kubectl get pods --namespace keycloak -l "app.kubernetes.io/name=keycloak-helm,app.kubernetes.io/instance=my-release" -o jsonpath="{.items[0].metadata.name}")
kubectl --namespace keycloak port-forward $POD_NAME 8080:8080
```

Then, open [http://localhost:8080](http://localhost:8080) in your browser.

## Key Features

### Realm Import vs. Database Provisioning

This chart offers two mutually exclusive methods to initialize your Keycloak instance:

*   **Realm Import (`keycloak.importRealm: true`):** This method imports a realm from a JSON file (`cmem.json`) located in a ConfigMap. This is useful for initializing a specific realm configuration on the first start.
*   **Database Provisioning (`postgres.provisioning.enabled: true`):** This method runs a Kubernetes Job to execute a SQL script (`keycloak_db.sql`) to provision the database. This is suitable for restoring a full database backup.

**Important:** You should only enable one of these options at a time.

### External Database

To use an external PostgreSQL database, set `postgres.internal` to `false` and provide the connection details:

```yaml
postgres:
  internal: false
  host: "my-external-postgres.example.com"
  port: 5432
  user: "keycloak"
  password: "mysecretpassword"
  database: "keycloak"
```

### Custom Theme

The eccenca theme can be enabled to customize the look and feel of Keycloak. This is enabled by default (`keycloak.eccencaTheme.enabled: true`) and uses an init container to copy the theme files into place.

## Uninstalling the Chart

To uninstall the `my-release` deployment:

```bash
helm uninstall my-release -n keycloak
```

## Configuration

The following table lists the configurable parameters of the Keycloak chart and their default values.

### General Settings

| Parameter | Description | Default |
| --- | --- | --- |
| `nameOverride` | Override the chart name. | `""` |
| `fullnameOverride` | Override the full name of the chart. | `""` |
| `imagePullSecrets` | List of image pull secrets for private registries. | `[]` |

### CMEM Configuration

| Parameter | Description | Default |
| --- | --- | --- |
| `cmem.hostname` | Hostname for CMEM-specific redirects and URIs. | `"cmem.docker.localhost"` |
| `cmem.cmemClientSecret` | Client secret for the cmem service account. Required for realm import. | `"changeme"` |
| `cmem.graphinsightsClientSecret` | Client secret for the graphinsights service account. Required for realm import. | `"changeme"` |

### Keycloak Configuration

| Parameter | Description | Default |
| --- | --- | --- |
| `keycloak.image.repository` | Keycloak image repository. | Not set (uses default) |
| `keycloak.image.tag` | Keycloak image tag. | Not set (uses default) |
| `keycloak.image.pullPolicy` | Keycloak image pull policy. | `Always` |
| `keycloak.setInitialAdmin` | Set initial admin user if not using import or restore. | `true` |
| `keycloak.adminUser` | Initial admin username. **Change for production!** | `admin` |
| `keycloak.adminPassword` | Initial admin password. **Change for production!** | `admin` |
| `keycloak.importRealm` | Enable realm import from ConfigMap on first start. Mutually exclusive with `postgres.provisioning.enabled`. | `false` |
| `keycloak.loglevel` | Log level for Keycloak. | `INFO` |
| `keycloak.cache` | Cache strategy. Possible values: `local`, `ispn`. | `local` |
| `keycloak.service.type` | Keycloak service type. | `ClusterIP` |
| `keycloak.service.port` | Keycloak service port. | `8080` |
| `keycloak.eccencaTheme.enabled` | Enable the eccenca custom theme. | `true` |
| `keycloak.eccencaTheme.image.repository` | Eccenca theme image repository. | Not set (uses default) |
| `keycloak.eccencaTheme.image.tag` | Eccenca theme image tag. | Not set (uses default) |
| `keycloak.eccencaTheme.image.pullPolicy` | Eccenca theme image pull policy. | `Always` |

### Ingress Configuration

| Parameter | Description | Default |
| --- | --- | --- |
| `ingress.enabled` | Enable ingress resource. | `false` |
| `ingress.className` | Ingress class name (e.g., `nginx`). | `"nginx"` |
| `ingress.annotations` | Additional annotations for the ingress resource. | `{}` |
| `ingress.hosts` | List of ingress hosts with paths. See values.yaml for structure. | `[]` |
| `ingress.tls` | TLS configuration for ingress. | `[]` |

### PostgreSQL Configuration

| Parameter | Description | Default |
| --- | --- | --- |
| `postgres.internal` | Use internal PostgreSQL database. Set to `false` for external database. | `true` |
| `postgres.image.repository` | PostgreSQL image repository. | Not set (uses default) |
| `postgres.image.tag` | PostgreSQL image tag. | Not set (uses default) |
| `postgres.image.pullPolicy` | PostgreSQL image pull policy. | `Always` |
| `postgres.user` | PostgreSQL username. | `keycloak` |
| `postgres.database` | PostgreSQL database name. | `keycloak` |
| `postgres.password` | PostgreSQL password. **Change for production!** | `yourKeycloakDatabasePassword398!` |
| `postgres.host` | External PostgreSQL host (when `internal: false`). | `""` |
| `postgres.port` | External PostgreSQL port (when `internal: false`). | `5432` |
| `postgres.persistentVolumeClaimName` | Use existing PVC instead of creating new one. | `""` |
| `postgres.persistenceSize` | Size of persistent volume for PostgreSQL data. | `1Gi` |
| `postgres.persistenceStorageClass` | Storage class for PVC. Uses cluster default if not set. | `""` |
| `postgres.persistenceAccessMode` | Access mode for PVC. | `ReadWriteOnce` |
| `postgres.provisioning.enabled` | Enable database provisioning from SQL dump. Mutually exclusive with `keycloak.importRealm`. | `false` |
| `postgres.provisioning.force` | Force re-provisioning (drops existing data). **Use with caution!** | `false` |
| `postgres.service.type` | PostgreSQL service type. | `ClusterIP` |

### Resource Management

| Parameter | Description | Default |
| --- | --- | --- |
| `resources.limits.cpu` | CPU limit for Keycloak container. | `1000m` |
| `resources.limits.memory` | Memory limit for Keycloak container. | `1024Mi` |
| `resources.requests.cpu` | CPU request for Keycloak container. | `200m` |
| `resources.requests.memory` | Memory request for Keycloak container. | `1024Mi` |

### Health Checks

| Parameter | Description | Default |
| --- | --- | --- |
| `livenessProbe.httpGet.path` | Path for liveness probe. | `/auth/health/live` |
| `livenessProbe.httpGet.port` | Port for liveness probe. | `9000` |
| `livenessProbe.initialDelaySeconds` | Initial delay for liveness probe. | `10` |
| `livenessProbe.timeoutSeconds` | Timeout for liveness probe. | `2` |
| `livenessProbe.periodSeconds` | Period for liveness probe. | `10` |
| `livenessProbe.failureThreshold` | Failure threshold for liveness probe. | `3` |
| `readinessProbe.httpGet.path` | Path for readiness probe. | `/auth/health/ready` |
| `readinessProbe.httpGet.port` | Port for readiness probe. | `9000` |
| `readinessProbe.initialDelaySeconds` | Initial delay for readiness probe. | `10` |
| `readinessProbe.timeoutSeconds` | Timeout for readiness probe. | `10` |
| `readinessProbe.periodSeconds` | Period for readiness probe. | `10` |
| `readinessProbe.failureThreshold` | Failure threshold for readiness probe. | `10` |

### Advanced Configuration

| Parameter | Description | Default |
| --- | --- | --- |
| `podAnnotations` | Additional annotations for Keycloak pods. | `{}` |
| `podLabels` | Additional labels for Keycloak pods. | `{}` |
| `podSecurityContext` | Security context for Keycloak pods. | `{}` |
| `securityContext` | Security context for Keycloak containers. | `{}` |
| `volumes` | Additional volumes for Keycloak deployment. | `[]` |
| `volumeMounts` | Additional volume mounts for Keycloak containers. | `[]` |
| `nodeSelector` | Node selector for Keycloak pods. | `{}` |
| `tolerations` | Tolerations for Keycloak pods. | `[]` |
| `affinity` | Affinity rules for Keycloak pods. | `{}` |

### Testing

| Parameter | Description | Default |
| --- | --- | --- |
| `tests.enabled` | Enable Helm test resources. | `false` |
| `tests.ingressControllerIP` | Ingress controller IP for local testing (kind/k3d). | `"10.43.82.133"` |

## Upgrading

To upgrade an existing Keycloak installation:

```bash
helm upgrade my-release ./keycloak-chart \
  --namespace keycloak \
  --reuse-values \
  --set keycloak.image.tag=NEW_VERSION
```

**Note:** Always review the changelog and test upgrades in a non-production environment first.

## Troubleshooting

### Keycloak Pod Not Starting

Check pod logs for errors:
```bash
kubectl logs -n keycloak -l app.kubernetes.io/name=keycloak-helm --tail=100
```

Common issues:
- Database connection failures: Verify PostgreSQL is running and credentials are correct
- Import/provisioning conflicts: Ensure only one initialization method is enabled
- Resource constraints: Check if pod has sufficient memory/CPU

### Database Connection Issues

Verify PostgreSQL connectivity:
```bash
# For internal database
kubectl exec -n keycloak -it deployment/my-release-keycloak-helm-postgres -- psql -U keycloak -d keycloak -c "SELECT version();"

# For external database
kubectl run -n keycloak psql-test --rm -it --restart=Never --image=postgres:17 -- psql -h EXTERNAL_HOST -U keycloak -d keycloak
```

### Realm Import Not Working

1. Verify `keycloak.importRealm` is set to `true`
2. Check that `cmem.hostname` is correctly configured
3. Ensure `postgres.provisioning.enabled` is `false` (mutually exclusive)
4. Check ConfigMap exists: `kubectl get configmap -n keycloak`

## Security Considerations

### Production Deployment Checklist

Before deploying to production:

- [ ] Change default admin credentials (`keycloak.adminUser` and `keycloak.adminPassword`)
- [ ] Update database password (`postgres.password`)
- [ ] Configure TLS/SSL for ingress
- [ ] Update client secrets (`cmem.cmemClientSecret`, `cmem.graphinsightsClientSecret`)
- [ ] Use external secret management (e.g., Kubernetes Secrets, HashiCorp Vault)
- [ ] Enable appropriate resource limits and requests
- [ ] Configure pod security contexts
- [ ] Review and configure network policies
- [ ] Use persistent storage with regular backups
- [ ] Configure appropriate ingress annotations for security headers

### Secrets Management

For production environments, consider using external secret management:

```yaml
# Example using external secrets
keycloak:
  adminPassword: ""  # Populated from external secret
postgres:
  password: ""       # Populated from external secret
```

Refer to your organization's secrets management solution (e.g., Sealed Secrets, External Secrets Operator, Vault) for implementation details.