# Keycloak Helm Chart

A Helm chart to deploy [Keycloak](https://www.keycloak.org/) in a non-operator manner on Kubernetes.

This chart provides a flexible and customizable way to set up Keycloak, including an optional internal PostgreSQL database, support for custom themes, and mechanisms for realm importation or database provisioning.

## Prerequisites

*   Kubernetes 1.12+
*   Helm 3.2.0+
*   An Ingress controller (e.g., [NGINX Ingress Controller](https://kubernetes.github.io/ingress-nginx/)) if `ingress.enabled` is `true`.

## Installation

To install the chart with the release name `my-release` into the `keycloak` namespace:

```bash
helm install my-release ./keycloak-chart --namespace keycloak --create-namespace
```

### Example Installation with Ingress

For a more realistic setup, you might want to enable ingress to expose Keycloak outside the cluster.

```bash
helm install my-release ./keycloak-chart \
  --namespace keycloak \
  --create-namespace \
  --set ingress.enabled=true \
  --set ingress.hosts[0].host=keycloak.example.com \
  --set ingress.hosts[0].paths[0].path=/ \
  --set keycloak.adminUser=admin \
  --set keycloak.adminPassword=mysecretpassword
```

### Local Development

For local development, you can use the provided `Taskfile.yaml`. First, create a `.env` file:

```bash
# .env
KUBE_CONFIG=k3d.cluster.config # Path to your kubeconfig file
GL_SLUG=local-testing          # Suffix for the namespace (kc-local-testing)
REVIEW_DOMAIN=docker.localhost # Domain for review apps
KEYCLOAK_HOSTNAME=kc.docker.localhost # Hostname for Keycloak
VALUE_FILE=kc-values.yaml      # Custom values file to use
```

Then, run the installation task:

```bash
task helm:install:local
```

## Accessing Keycloak

Once the chart is deployed, you can access the Keycloak UI.

If you enabled ingress, the URL will be printed in the installation notes. You can also find it with:

```bash
echo "https://$(kubectl get ingress -n keycloak my-release-keycloak-helm -o jsonpath='{.spec.rules[0].host}')"
```

If you are not using ingress, you can access Keycloak via port-forwarding:

```bash
export POD_NAME=$(kubectl get pods --namespace keycloak -l "app.kubernetes.io/name=keycloak-helm,app.kubernetes.io/instance=my-release" -o jsonpath="{.items[0].metadata.name}")
kubectl --namespace keycloak port-forward $POD_NAME 8080:8080
```

Then, open [http://localhost:8080](http://localhost:8080) in your browser.

## Key Features

### Realm Import vs. Database Provisioning

This chart offers two mutually exclusive methods to initialize your Keycloak instance:

*   **Realm Import (`keycloak.importRealm: true`):** This method imports a realm from a JSON file (`cmem.json`) located in a ConfigMap. This is useful for initializing a specific realm configuration on the first start.
*   **Database Provisioning (`postgres.provisioning.enabled: true`):** This method runs a Kubernetes Job to execute a SQL script (`keycloak_db.sql`) to provision the database. This is suitable for restoring a full database backup.

**Important:** You should only enable one of these options at a time.

### External Database

To use an external PostgreSQL database, set `postgres.internal` to `false` and provide the connection details:

```yaml
postgres:
  internal: false
  host: "my-external-postgres.example.com"
  port: 5432
  user: "keycloak"
  password: "mysecretpassword"
  database: "keycloak"
```

### Custom Theme

The eccenca theme can be enabled to customize the look and feel of Keycloak. This is enabled by default (`keycloak.eccencaTheme.enabled: true`) and uses an init container to copy the theme files into place.

## Uninstalling the Chart

To uninstall the `my-release` deployment:

```bash
helm uninstall my-release -n keycloak
```

## Configuration

The following table lists the configurable parameters of the Keycloak chart and their default values.

| Parameter | Description | Default | 
| --- | --- | --- |
| `nameOverride` | Override the chart name. | `""` |
| `fullnameOverride` | Override the full name of the chart. | `""` |
| `cmem.cmemClientSecret` | Client secret for the cmem service account. Only needed for realm import. | `""` |
| `keycloak.image.repository` | Keycloak image repository. | `quay.io/keycloak/keycloak` |
| `keycloak.image.tag` | Keycloak image tag. | `26.3.4-0` |
| `keycloak.image.imagePullPolicy` | Keycloak image pull policy. | `Always` |
| `keycloak.setInitialAdmin` | Set initial admin user if not using import or restore. | `true` |
| `keycloak.adminUser` | Initial admin username. | `admin` |
| `keycloak.adminPassword` | Initial admin password. | `admin` |
| `keycloak.importRealm` | Enable realm import from a ConfigMap on first start. | `false` |
| `keycloak.restoreFromBackup` | Restore from a database backup using the provisioning job. | `true` |
| `keycloak.eccencaTheme.enabled` | Enable the eccenca theme. | `true` |
| `keycloak.eccencaTheme.image.repository` | Eccenca theme image repository. | `docker-registry.eccenca.com/eccenca-keycloak-theme` |
| `keycloak.eccencaTheme.image.tag` | Eccenca theme image tag. | `v2.4.0` |
| `keycloak.eccencaTheme.image.imagePullPolicy` | Eccenca theme image pull policy. | `Always` |
| `keycloak.service.type` | Keycloak service type. | `ClusterIP` |
| `keycloak.service.port` | Keycloak service port. | `8080` |
| `ingress.enabled` | Enable ingress. | `false` |
| `ingress.className` | Ingress class name. | `""` |
| `ingress.annotations` | Ingress annotations. | `{}` |
| `ingress.hosts` | Ingress hosts. Example: `[{host: keycloak.example.com, paths: [{path: /, pathType: ImplementationSpecific}]}]` | `[]` |
| `ingress.tls` | Ingress TLS configuration. | `[]` |
| `postgres.internal` | Enable internal PostgreSQL database. | `true` |
| `postgres.image.repository` | PostgreSQL image repository. | `postgres` |
| `postgres.image.tag` | PostgreSQL image tag. | `17` |
| `postgres.image.imagePullPolicy` | PostgreSQL image pull policy. | `IfNotPresent` |
| `postgres.user` | PostgreSQL username. | `keycloak` |
| `postgres.database` | PostgreSQL database name. | `keycloak` |
| `postgres.password` | PostgreSQL password. | `yourKeycloakDatabasePassword398!` |
| `postgres.persistentVolumeClaimName` | Name of the persistent volume claim. | `""` |
| `postgres.persistenceSize` | PostgreSQL data persistence size. | `1Gi` |
| `postgres.persistenceStorageClass` | PostgreSQL data persistence storage class. | `""` |
| `postgres.persistenceAccessMode` | PostgreSQL data persistence access mode. | `ReadWriteOnce` |
| `postgres.provisioning.enabled` | Enable database provisioning from `keycloak_db.sql`. | `false` |
| `postgres.provisioning.force` | Force database provisioning on every start. | `false` |
| `postgres.host` | External PostgreSQL host. | `""` |
| `postgres.port` | External PostgreSQL port. | `5432` |
| `postgres.service.type` | PostgreSQL service type. | `ClusterIP` |